%% Replication code for: 
% COMMON FACTOR OF COMMODITY PRICES (2021)
% by Simona Delle Chiaie, Laurent Ferrara and Domenico Giannone
% Corresponding author: S. Delle Chiaie 
% (email:simona.dellechiaie@ecb.europa.eu)
% This program has been written for the MATLAB version R2020a
% The software can be freely used in applications. 
% Users are kindly requested to add acknowledgements to published work and 
% to cite the above reference in any resulting publications.

% First version: Paris, 13 April 2015
% last update: Frankfurt am Main, July 2021

clear;
clc;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Section 1. set path and folders
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

rootfolder= pwd;
dataset = strcat(rootfolder,'\dataset\'); % data path
    if ~exist(dataset, 'dir')
    mkdir(dataset);
    end
functions = strcat(rootfolder,'\functions\'); % functions
    if ~exist(functions, 'dir')
    mkdir(functions);
    end
output = strcat(rootfolder,'\output\'); % results
    if ~exist(output, 'dir')
     mkdir(output);
    end
addpath(dataset); addpath(functions); addpath(output); % output

namesave = strcat('sav_',datestr(today)); % date

%% Section 2: Load data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[DATA,TEXT]= xlsread('Commodities_IMF_data.xlsx','data'); 
Time = datenum((TEXT(2:end,1)));
X = DATA;
Mnem = TEXT(1,2:end)';
[DATA,TEXT]= xlsread('Commodities_IMF_data.xlsx','legend');
Unit = TEXT(2:end,2);
Description = TEXT(2:end,3);
BlockNames = TEXT(1,4:end-1)';
Blocks = DATA(:,1:end-1);
Global = DATA(:,1);
Select = DATA(:,end);
[DATA,TEXT]= xlsread('Commodities_IMF_data.xlsx','weights');
Weights = DATA; 
clear DATA TEXT   
disp ('Section 2: Data loaded')


%% Section 3: Create variables and select data for estimation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

X = X(:,Select==1);
Global = Global(Select==1);
Description = Description(Select==1);
Blocks = Blocks(Select==1,:);
Unit = Unit(Select==1);
Mnem = Mnem(Select==1);
Xdfm = X(:,Global==1); 
Xind = X(:,Global==0);
MnemDfm = Mnem(Global==1);
BlockDfm = Blocks(Global==1,:);
DescriptionDfm = Description(Global==1);

x = diff(log(Xdfm))*100;
% y = diff(log(Xind))*100;


TimeDfm = Time(2:end,:);
DatesDfm = datevec(TimeDfm);

% select estimation sample
BegEstY = 1981; BegEstM = 1;
EndEstY = 2020; EndEstM = 3; 

TbegEst = find(DatesDfm(:,1)==BegEstY & DatesDfm(:,3)==BegEstM); 
TendEst = find(DatesDfm(:,1)==EndEstY & DatesDfm(:,3)==EndEstM);

xest = x(TbegEst:TendEst,:);
DatesEst = DatesDfm(TbegEst:TendEst,:);
TimeEst = datenum(DatesEst);

[t,m] = size(xest);

xin = xest;

xin(xin==0) = NaN;

% check missing data
SerOK = (sum(isnan(xin))<t/4); 

xin= xin(:,SerOK);
Mnemest = MnemDfm(SerOK,1);
Blockest = BlockDfm(SerOK,:);
% w = Weights(SerOK,:);
Descriptionest = DescriptionDfm(SerOK,1);
[t,m] = size(xin);

clear X Time TimeDfm load_data 
disp ('Section 3: Create variables and select data for estimation ends')

%% Section 4: ESTIMATION DFM WITH BLOCKS USING EM ALGORITHM
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

r = 1; % number of common factors
Par.r = ones(size(Blockest,2),1); Par.r(1) = r; %Number of block factors
Par.p = 1; % Lags in the VAR on the factors
Par.nQ = 0; %n. of quarterly variables, it is necessary for the model to understand the positions
lags = 5; % necessary to find the position
Par.blocks = [Blockest];
Par.thresh = 1e-4;
Par.max_iter = 1000;

disp ('Estimation starts')
Res_in = EM_DFM_SS_block_idioQARMA_restrMQ(xin,Par);

clear Select_blocks TbegEst TendEst 

disp ('Section 4: Estimation completed')

%% Section 5: Collect results
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Factor loadings
C = [Res_in.C(:,1:r) Res_in.C(:,r*lags+1:lags:(size(Par.blocks,2)+(r-1))*lags)];

% Global and block factors and idio
F_t = [Res_in.F(:,1:r) Res_in.F(:,r*lags+1:lags:(size(Par.blocks,2)+(r-1))*lags)];
e_t = Res_in.F(:,(lags*size(Par.blocks,2)+1:end));

% recalculate standard deviation and mean
Mx = nanmean(xin);
Sx = nanstd(xin);
xNaN = (xin-repmat(Mx,t,1))./repmat(Sx,t,1);

x_F= F_t*C';

for i=1:size(Par.blocks,2)
x_f(:,:,i)= F_t(:,i)*C(:,i)';
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Figure 6 Historical decomposition of commodity prices 2000-2008
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

set(0,'DefaultAxesTitleFontWeight','normal');


first_y = 2000; first_m = 1; %% Dates of the beginning of chart
chart_first = find((DatesEst(:,1)==first_y) & (DatesEst(:,3)==first_m));

last_y = 2008; last_m = 7;   %% Dates of the end of the chart
chart_last = find((DatesEst(:,1)==last_y) & (DatesEst(:,3)==last_m));

% Define commodity prices to decompose
a = find(strcmp(Mnemest, 'Copper')); 
b = find(strcmp(Mnemest, 'Brent Crude Oil')); 
c = find(strcmp(Mnemest, 'Nickel')); 
d = find(strcmp(Mnemest, 'Maize')); 

oil_f = cumsum([x_f(chart_first:chart_last,b,1) x_f(chart_first:chart_last,b,10) x_f(chart_first:chart_last,b,11) e_t(chart_first:chart_last,b)]);
oil_f = [oil_f(:,1) sum(oil_f(:,2:end-1),2) oil_f(:,end)]*Sx(b);
nick_f = cumsum([x_f(chart_first:chart_last,c,1) x_f(chart_first:chart_last,c,2) x_f(chart_first:chart_last,c,6) x_f(chart_first:chart_last,c,8) e_t(chart_first:chart_last,c)]);
nick_f = [nick_f(:,1) sum(nick_f(:,2:end-1),2) nick_f(:,end)]*Sx(c);
corn_f = cumsum([x_f(chart_first:chart_last,d,1) x_f(chart_first:chart_last,d,2) x_f(chart_first:chart_last,d,3) x_f(chart_first:chart_last,d,4) e_t(chart_first:chart_last,d)]);
corn_f = [corn_f(:,1) sum(corn_f(:,2:end-1),2) corn_f(:,end)]*Sx(18);
copp_f = cumsum([x_f(chart_first:chart_last,a,1) x_f(chart_first:chart_last,a,2) x_f(chart_first:chart_last,a,6) x_f(chart_first:chart_last,a,8) e_t(chart_first:chart_last,a)]);
copp_f = [copp_f(:,1) sum(copp_f(:,2:end-1),2) copp_f(:,end)]*Sx(a);
% 
% 
time=(2000+1/12:1/12:2008+7/12)';
% 
figure
subplot(2,2,1)
oil_pos = oil_f(1:end,:);
oil_pos(oil_pos>0) = 0;
oil_neg = oil_f(1:end,:);
oil_neg(oil_neg<0) = 0;
hold on
clear a b c d
% 
a = bar(time,oil_pos,'stacked','EdgeColor','none');
set(a(1),'FaceColor','b')
set(a(2),'FaceColor','r')
set(a(3),'FaceColor',[0.5 0.5 0.5])
b = bar(time,oil_neg,'stacked','EdgeColor','none');
set(b(1),'FaceColor','b')
set(b(2),'FaceColor','r')
set(b(3),'FaceColor',[0.5 0.5 0.5])
plot(time, cumsum(xNaN(chart_first:chart_last,21))*Sx(21),'k','LineWidth',2);
set(gca,'FontSize',12,'xlim',[2000 2008.7])
set(gca,'XTick',2000:24/12:2008)
set(gca,'XTickLabel',{'2000','2002','2004','2006','2008'})
title('Brent oil')
hold off
clear a b

subplot(2,2,2)
copp_pos = copp_f(1:end,:);
copp_pos(copp_pos>0) = 0;
copp_neg = copp_f(1:end,:);
copp_neg(copp_neg<0) = 0;
hold on
a = bar(time,copp_pos,'stacked','EdgeColor','none');
set(a(1),'FaceColor','b')
set(a(2),'FaceColor','r')
set(a(3),'FaceColor',[0.5 0.5 0.5])
b = bar(time,copp_neg,'stacked','EdgeColor','none');
set(b(1),'FaceColor','b')
set(b(2),'FaceColor','r')
set(b(3),'FaceColor',[0.5 0.5 0.5])
plot(time, cumsum(xNaN(chart_first:chart_last,9))*Sx(9),'k','LineWidth',2);
set(gca,'FontSize',12,'xlim',[2000 2008.7])
set(gca,'XTick',2000:24/12:2008)
set(gca,'XTickLabel',{'2000','2002','2004','2006','2008'})
title('Copper')
hold off
clear a b

subplot(2,2,3)
corn_pos = corn_f(1:end,:);
corn_pos(corn_pos>0) = 0;
corn_neg = corn_f(1:end,:);
corn_neg(corn_neg<0) = 0;
hold on
a = bar(time,corn_pos,'stacked','EdgeColor','none');
set(a(1),'FaceColor','b')
set(a(2),'FaceColor','r')
set(a(3),'FaceColor',[0.5 0.5 0.5])
b = bar(time,corn_neg,'stacked','EdgeColor','none');
set(b(1),'FaceColor','b')
set(b(2),'FaceColor','r')
set(b(3),'FaceColor',[0.5 0.5 0.5])
plot(time, cumsum(xNaN(chart_first:chart_last,18))*Sx(18),'k','LineWidth',2);
set(gca,'FontSize',12,'xlim',[2000 2008.7])
set(gca,'XTick',2000:24/12:2008)
set(gca,'XTickLabel',{'2000','2002','2004','2006','2008'})
title('Corn')
hold off
clear a b


subplot(2,2,4)
nick_pos = nick_f(1:end,:);
nick_pos(nick_pos>0) = 0;
nick_neg = nick_f(1:end,:);
nick_neg(nick_neg<0) = 0;
hold on
a = bar(time, nick_pos,'stacked','EdgeColor','none');
set(a(1),'FaceColor','b')
set(a(2),'FaceColor','r')
set(a(3),'FaceColor',[0.5 0.5 0.5])
b = bar(time, nick_neg,'stacked','EdgeColor','none');
set(b(1),'FaceColor','b')
set(b(2),'FaceColor','r')
set(b(3),'FaceColor',[0.5 0.5 0.5])
plot(time, cumsum(xNaN(chart_first:chart_last,20))*Sx(20),'k','LineWidth',2);

legend([a,b],{'Global factor','Block factors','Idiosyncratic'},'Location','Best','FontSize',9);
legend('boxoff')
set(gca,'FontSize',12,'xlim',[2000 2008.7])
set(gca,'XTick',2000:24/12:2008.7)
set(gca,'XTickLabel',{'2000','2002','2004','2006','2008'})
title('Nickel')
hold off
clear a b
% 
filename = fullfile(output, 'Figure6');
savefig(filename);
print(filename,'-dpng');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Figure 7 Historical decomposition of oil prices 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% 1. Saudia Arabia 1986 cartel collapse 

oil = find(strcmp(Mnemest, 'Brent Crude Oil')); 

first_y = 1985; first_m = 12; 
chart_first = find((DatesEst(:,1)==first_y) & (DatesEst(:,3)==first_m));

last_y = 1986; last_m = 8;  
chart_last = find((DatesEst(:,1)==last_y) & (DatesEst(:,3)==last_m));


z = zeros(chart_last-chart_first+1,1);
oil_f = cumsum([x_f(chart_first:chart_last,oil,1) x_f(chart_first:chart_last,oil,10) x_f(chart_first:chart_last,oil,11) e_t(chart_first:chart_last,oil)]);
oil_f = [oil_f(:,1) sum(oil_f(:,2:end-1),2) oil_f(:,end)]*Sx(oil);


time =(1985+12/12:1/12:1986+8/12)';

figure
subplot(2,2,1)

oil_pos = oil_f;
oil_pos(oil_pos>0) = 0;
oil_neg = oil_f;
oil_neg(oil_neg<0) = 0;
hold on
a = bar(oil_pos,'stacked','EdgeColor','none','BarWidth',0.6);
set(a(1),'FaceColor','b')
set(a(2),'FaceColor','r')
set(a(3),'FaceColor',[0.5 0.5 0.5])

b = bar(oil_neg,'stacked','EdgeColor','none','BarWidth',0.6);
set(b(1),'FaceColor','b')
set(b(2),'FaceColor','r')
set(b(3),'FaceColor',[0.5 0.5 0.5])

plot(cumsum(xNaN(chart_first:chart_last,oil))*Sx(oil),'k','LineWidth',3);
title('The oil price fall in 1986')
set(gca,'FontSize',12,'XTick',1:4:10)
set(gca,'XTickLabel',{'Dec85','Apr86','Aug86'})

clear oil_f oil_pos oil_neg a b
hold off

% 2. Gulf war 90-91
first_y = 1990; first_m = 7; 
chart_first = find((DatesEst(:,1)==first_y) & (DatesEst(:,3)==first_m));
last_y = 1991; last_m = 3;   
chart_last = find((DatesEst(:,1)==last_y) & (DatesEst(:,3)==last_m));
time=(1990+7/12:1/12:1991+3/12)';

subplot(2,2,2)

oil_f = cumsum([x_f(chart_first:chart_last,oil,1) x_f(chart_first:chart_last,oil,10) x_f(chart_first:chart_last,oil,11) e_t(chart_first:chart_last,oil)]);
oil_f = [oil_f(:,1) sum(oil_f(:,2:end-1),2) oil_f(:,end)]*Sx(oil);
oil_pos = oil_f;
oil_pos(oil_pos>0) = 0;
oil_neg = oil_f;
oil_neg(oil_neg<0) = 0;
hold on
a = bar(oil_pos,'stacked','EdgeColor','none','BarWidth',0.6);
set(a(1),'FaceColor','b')
set(a(2),'FaceColor','r')
set(a(3),'FaceColor',[0.5 0.5 0.5])
b = bar(oil_neg,'stacked','EdgeColor','none','BarWidth',0.6);
set(b(1),'FaceColor','b')
set(b(2),'FaceColor','r')
set(b(3),'FaceColor',[0.5 0.5 0.5])
plot(cumsum(xNaN(chart_first:chart_last,oil))*Sx(oil),'k','LineWidth',3);
set(gca,'FontSize',12,'XTick',1:4:10)
set(gca,'XTickLabel',{'Jul90','Nov90','Mar91'})
title('The Gulf War in 1990-91')
hold off
clear oil_f oil_pos oil_neg

% 3. Great recession 2008-2009
first_y = 2008; first_m = 8; 
chart_first = find((DatesEst(:,1)==first_y) & (DatesEst(:,3)==first_m));
last_y = 2009; last_m = 2;   
chart_last = find((DatesEst(:,1)==last_y) & (DatesEst(:,3)==last_m));

time=(2008+8/12:1/12:2009+2/12)';

subplot(2,2,3)

oil_f = cumsum([x_f(chart_first:chart_last,oil,1) x_f(chart_first:chart_last,oil,10) x_f(chart_first:chart_last,oil,11) e_t(chart_first:chart_last,oil)]);
oil_f = [oil_f(:,1) sum(oil_f(:,2:end-1),2) oil_f(:,end)]*Sx(oil);
oil_pos = oil_f;
oil_pos(oil_pos>0) = 0;
oil_neg = oil_f;
oil_neg(oil_neg<0) = 0;
hold on
a = bar(oil_pos,'stacked','EdgeColor','none','BarWidth',0.6);
set(a(1),'FaceColor','b')
set(a(2),'FaceColor','r')
set(a(3),'FaceColor',[0.5 0.5 0.5])
b = bar(oil_neg,'stacked','EdgeColor','none','BarWidth',0.6);
set(b(1),'FaceColor','b')
set(b(2),'FaceColor','r')
set(b(3),'FaceColor',[0.5 0.5 0.5])
plot(cumsum(xNaN(chart_first:chart_last,oil))*Sx(oil),'k','LineWidth',3);
set(gca,'FontSize',12,'XTick',1:3:7)
set(gca,'XTickLabel',{'Aug08','Dec08','Feb09'})
xlim([0 8])
ylim([-200 50])
title('The Great Recession')
hold off
clear oil_f oil_pos oil_neg a b


% 4. The oil price fall in 2014
first_y = 2014; first_m = 7; 
chart_first = find((DatesEst(:,1)==first_y) & (DatesEst(:,3)==first_m));
last_y = 2015; last_m = 12;   
chart_last = find((DatesEst(:,1)==last_y) & (DatesEst(:,3)==last_m));

oil_f = cumsum([x_f(chart_first:chart_last,oil,1) x_f(chart_first:chart_last,oil,10) x_f(chart_first:chart_last,oil,11) e_t(chart_first:chart_last,oil)]);
oil_f = [oil_f(:,1) sum(oil_f(:,2:end-1),2) oil_f(:,end)]*Sx(oil);
oil_pos = oil_f;
oil_pos(oil_pos>0) = 0;
oil_neg = oil_f;
oil_neg(oil_neg<0) = 0;

subplot(2,2,4)
a = bar(oil_pos,'stacked','EdgeColor','none','BarWidth',0.8);
set(a(1),'FaceColor','b')
set(a(2),'FaceColor','r')
set(a(3),'FaceColor',[0.5 0.5 0.5])
hold on
b = bar(oil_neg,'stacked','EdgeColor','none','BarWidth',0.8);
set(b(1),'FaceColor','b')
set(b(2),'FaceColor','r')
set(b(3),'FaceColor',[0.5 0.5 0.5])
plot(cumsum(xNaN(chart_first:chart_last,oil))*Sx(oil),'k','LineWidth',3);
xlim([1 19])
ylim([-200 50])
% 2014.7::2008.7
legend([a,b],{'Global factor','Block factors','Idiosyncratic'},'Location','Best','FontSize',9);
legend('boxoff')
set(gca,'FontSize',12,'XTick',3:7:19)
set(gca,'XTickLabel', {'Sep14','Apr15','Nov15'})
title('The oil price fall in 2014-15')
hold off
clear oil_f oil_pos oil_neg a b

filename = fullfile(output, 'Figure7');
savefig(filename);
print(filename,'-dpng');

